/*
  TODO:
  - *text*   -> em
  - **text** -> strong
  - links
  - inline images
  - parse line-elements (such as tags)
  
   BUGS:
   - starting a line with "* * text" creates a double list with "<li> text"
     on a line all by itself, incorrectly indented.
*/
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>

#define nil 1
#define ul  2
#define li  3
#define ol  4
#define dl  5
#define dt  6
#define dd  7
#define em  8
#define strong 9
#define maxcode 10

char *code[maxcode] = {"", "", "ul", "li", "ol", "dl", "dt", "dd", "em", "strong"};

typedef struct {
  int t;                               /* type */
  int i;                               /* indent level */
} stack_arr_t;

#define max_stack_depth 128

typedef struct {
  int n;                               /* number of elements on stack */
  stack_arr_t st[max_stack_depth];              /* stack */
} stack_t;

void print_indent(int i,               /*  indent */
                  int o                /* offset */
                  )
{
  if (i > 0) {
    if (o == 0) {
      printf("."); 
    } else {
      printf(" ");
    }
  }
  for (; i>1; i--) {
    printf(" ");
  }
}

void add_to_stack(stack_t *st,         /* stack */
                  int t,               /* type */
                  int i                /* indent */
                  )
{
  if (st->n <= max_stack_depth) {
    st->st[st->n].t = t;
    st->st[st->n].i = i;
    st->n++;
  } else {
    errno = ENOMEM;
    perror("in function add_to_stack()");
    exit(ENOMEM);
  }
}

void rewind_stack(char p[],            /* prefix */
                  stack_t *st,          /* stack */
                  int d                /* depth */
                  )
{
  for (; st->st[st->n-1].i>d; st->n--) {
    switch (st->st[st->n-1].t) {
    case ul:
    case ol:
    case dl:
    case dd:
      printf("%s", p);
      print_indent(st->st[st->n-1].i, 0);
      printf("</%s>\n", code[st->st[st->n-1].t]);
      break;
    case dt:
      printf("</%s>\n", code[st->st[st->n-1].t]);
      break;
    case li:
      /* printf("%s", p);
         print_indent(st->st[st->n-1].i-2, 0);
         printf("</%s>\n", code[st->st[st->n-1].t]); */
      break;
    case em:
    case strong:
      /* if this is called here we have a parse error! */
      printf("%s", p);
      print_indent(st->st[st->n-1].i-2, 0);
      printf("</%s><!-- ERROR! -->\n", code[st->st[st->n-1].t]); 
      break;
    default:
      printf("t=%d, i=%d\n", st->st[st->n-1].t, st->st[st->n-1].i);
      break;
    }
  }
}

void rewind_stack_by_type(char p[],    /* prefix */
                          stack_t *st, /* stack */
                          int t        /* type */
                          )
{
  while (1) {
    switch (st->st[st->n-1].t) {
    case ul:
    case ol:
    case dl:
    case dd:
      printf("%s", p);
      print_indent(st->st[st->n-1].i, 0);
      printf("</%s>\n", code[st->st[st->n-1].t]);
      break;
    case dt:
      printf("</%s>\n", code[st->st[st->n-1].t]);
      break;
    case li:
      /* printf("%s", p);
         print_indent(st->st[st->n-1].i-2, 0);
         printf("</%s>\n", code[st->st[st->n-1].t]); */
      break;
    case em:
    case strong:
      printf("</%s>", code[st->st[st->n-1].t]); 
      break;
    default:
      printf("t=%d, i=%d\n", st->st[st->n-1].t, st->st[st->n-1].i);
      break;
    }
    st->n--;
    if ((st->n == 0) || (st->st[st->n].t == t)) {
      goto out;
    }
  } /* end while forever */
 out:
  return;
}

/*
  Match the following:
  [\n ]_text_[\n ]           -> ul
  [\n ]\*text\*[\n ]         -> em
  [\n ]\*\*text\*\*[\n ]     -> bf
  [\n ]<url>[\n ]            -> url
  [\n ]<<url>link text>[\n ] -> url with link text.
*/

int parse_line(char    p[],                 /* prefix */
               int     o,                   /* offset */
               char    s[],                 /* string */
               stack_t *st                  /* stack */
               )
{
  int i;                               /* intendation, index */
  int j,k;                             /* loop control */
  int n;                               /* original strlen(s) */
  char *url;
  char *tag;

  n = strlen(s);
  for (i=0; i<n; i++) {
    switch (s[i]) {
    case '*':                          /* em or strong */
      /* prefixed by [\n ] */
      if ((i == 0) || (s[i-1] == ' ')) {
        switch (s[i+1]) {
        case '*':
          switch (s[i+2]) {
          case '*':
            /* three or more * in a row: ***
               maybe make this a <hr>??? */
            printf("***");
            i += 2;
            break;
          case 0:
          case ' ':
            /* followed by [\n ] */
            printf("**");
            i++;
          default:
            /* new strong */
            add_to_stack(st, strong, i+o);
            printf("<strong>");
            i++;                       /* to skip the second * */
          }
          break;
        case 0:
        case ' ':
          /* followed by [\n ] */
          printf("*");
          break;
        default:
          /* new em */
          add_to_stack(st, em, i+o);
          printf("<em>");
        } /* end switch */
      } /* end if start */
      else if (st->st[st->n-1].t == em) {
        switch (s[i+1]) {
        case 0:
        case ' ':
          /* close em */
          rewind_stack_by_type(p, st, em);
          break;
        default:
          printf("*");
        }
      } /* end if em */
      else if (st->st[st->n-1].t == strong) {
        if ((s[i+1] == '*') && ((s[i+2] == 0) || (s[i+2] == ' '))) {
          rewind_stack_by_type(p, st, strong);
          i++;
        } else {
          printf("*");
        }
      } /* end if strong */
      break;
    case '<':                          /* url */
      for (j=0, k=0; (s[j] != 0) && (s[j] != '>');) {j++;}
      if (s[j] != '>') {
        /* no closing bracket, can't be an url */
        printf("<");
      } else {
        url = &s[1];
        if (url[0] == '<') {
          url++;
          for (k=j+1; (s[k] != 0) && (s[k] != '>');) {k++;}
          if (s[k] != '>') {
            /* no closing bracket, can't be an url */
            printf("<");
          } else {
            tag = &s[j+1];
            s[k] = 0;
          } /* end if proper url with tag */
        } /* end if url with tag */
        else {
          tag = url;
        }
        s[j] = 0;
        printf("<a href=\"%s\">", url);
        parse_line(p,o,tag,st);
        printf("</a>");
        i = (j>k?j:k);
      } /* end if url */
      break;
    default:
      printf("%c", s[i]);
    } /* end switch */
  } /* end for i */
  return 0;
}

int parse(char    p[],                 /* prefix */
          int     o,                   /* offset */
          char    s[],                 /* string */
          stack_t *st                  /* stack */
          )
{
  int i=0;                             /* indentation */
  int j;                               /* loop control */

  /* find indentation */
  if ((s[0] == ' ') || (s[0] == '.')) {
    i = 1;
    while (s[i] == ' ') {
      i++;
    }
  }
  if (i+o < st->st[st->n-1].i) {
    rewind_stack(p, st, i+o);
  }
  switch (s[i]) {
  case 0:
    /* "empty" line */
    printf("%s", p);
    print_indent(i,o);
    printf("%s\n", &s[i]);    
    break;
  case '-':
  case '*':
  case '+':
    /* an unordered list starts with one of [-+*] followed by a space */
    if (s[i+1] == ' ') {
      if (st->st[st->n-1].t != ul) {
        rewind_stack(p, st, i+o-1);
        /* new ul */
        add_to_stack(st, ul, i+o);
        printf("%s", p);
        print_indent(i, o);
        printf("<ul>\n");
      }
      /* new li */
      add_to_stack(st, li, i+2+o);
      printf("%s", p);
      print_indent(i, o);
      printf("<li> ");
      parse("", i+2+o, &s[i+2], st);
      break;
    } else {
      goto out;
    }
  case ':':
    /* a description list starts with ':.*:[ \n] ' */
    if (s[i+1] != ':') {
      for (j=2; (s[i+j] != 0) && !((s[i+j] == ':') && ((s[i+j+1] == 0) || (s[i+j+1] == ' ')));) {j++;}
      if ((s[i+j] != ':') || ((s[i+j+1] != ' ') && (s[i+j+1] != 0))) {goto out;}
      if (st->st[st->n-1].t != dl) {
        rewind_stack(p, st, i+o-1);
        /* new ol */
        add_to_stack(st, dl, i+o);
        printf("%s", p);
        print_indent(i, o);
        printf("<dl>\n");
      }
      /* new li */
      /* add_to_stack(st, dt, i+1+o);
         Don't --- we're gonna close it soon anyway */
      printf("%s", p);
      print_indent(i+1, o);
      printf("<dt>");
      s[j+i]=0;
      parse_line(p,o,&s[i+1],st);
      printf("</dt>\n");
      add_to_stack(st, dd, i+1+o);
      printf("%s", p);
      print_indent(i+1,o);
      printf("<dd>");
      parse("", i+j+o+2, &s[i+j+2], st);
      break;
    } else {
      goto out;
    }      
  case '(':
    /* an ordered list does e.g. start with '(X) ', where X is one of [a-z], [A-Z], [0-9]+, or, not implemented, [mcxvi] or [MCXVI] */
    if ((s[i+2] == ')') && (s[i+3] == ' ') && ((s[i+1] == '#') || isalnum(s[i+1]))) {
      if (st->st[st->n-1].t != ol) {
        rewind_stack(p, st, i+o-1);
        /* new ol */
        add_to_stack(st, ol, i+o);
        printf("%s", p);
        print_indent(i, o);
        printf("<ol>\n");
      }
      /* new li */
      add_to_stack(st, li, i+4+o);
      printf("%s", p);
      print_indent(i, o);
      printf("<li> ");
      parse("", i+4+o, &s[i+4], st);
      break;
    } else {
      goto out;
    }      
  default:
    /* some things need to be decided on something other than the first character */
    /* ol can e.g. be one of '[#:alnum:][.)] ' */
    if (((s[i+1] == ')') || (s[i+1] == '.')) && (s[i+2] == ' ') && ((s[i] == '#') || isalnum(s[i]))) {
      if (st->st[st->n-1].t != ol) {
        rewind_stack(p, st, i+o-1);
        /* new ol */
        add_to_stack(st, ol, i+o);
        printf("%s", p);
        print_indent(i, o);
        printf("<ol>\n");
      }
      /* new li */
      add_to_stack(st, li, i+3+o);
      printf("%s", p);
      print_indent(i, o);
      printf("<li> ");
      parse("", i+3+o, &s[i+3], st);
      break;
    }     
    /* ol can e.g. be one of '[:digit:]+[.)] ' */
    if (isdigit(s[i])) {
      for (j=1; isdigit(s[j+i]); ) {j++;}
      if ((s[j+i+1] == ' ') && ((s[j+i] == '.') || (s[j+i] == ')'))) {
        if (st->st[st->n-1].t != ol) {
          rewind_stack(p, st, i+o-1);
          /* new ol */
          add_to_stack(st, ol, i+o);
          printf("%s", p);
          print_indent(i, o);
          printf("<ol>\n");
        }
        /* new li */
        add_to_stack(st, li, i+j+2+o);
        printf("%s", p);
        print_indent(i, o);
        printf("<li> ");
        parse("", i+j+2+o, &s[i+j+2], st);
        break;
      }
    }     
    goto out;
  }
  return 0;
 out:
  rewind_stack(p, st, i+o);
  if ((i+o == 0) && (st->n > 1)) {
    /* back to square one - clean up all */
    rewind_stack(p, st, -1);
  }      
  printf("%s", p);
  print_indent(i,o);
  parse_line(p,o,&s[i],st);
  printf("\n");
  return 0;
}

int main()
{
  char s[1024]  = {0};                 /* string */
  stack_t st    = {0};                 /* stack */
  char p[1024]  = {0};                 /* prefix */
  int i;

  st.n=0;
  add_to_stack(&st, nil, -1);
  
  while (fgets(s, 1023, stdin)) {
    /* remove newline */
    s[strlen(s)-1] = 0;
    /* We only deal with indented comment lines, starting with a '|' */
    if (s[0] == '|') {
      /* remove prefix, consisting of '|' and N ' ' */
      if (p[0] == 0) {
        p[0] = s[0];
        i = 1;
        while (s[i] == ' ') {
          p[i] = s[i];
          i++;
        }
        p[i] = 0;
      }
      parse(p, 0, &s[strlen(p)], &st);
      
    } else {
      rewind_stack(p, &st, -1);
      p[0] = 0;
      /* copy line unmodified */
      printf("%s\n", s);
    }
  }
  
  return 0;
}
